<?php

/**
 * This is the model class for table "{{warehouse_transfer_history}}".
 *
 * The followings are the available columns in table '{{warehouse_transfer_history}}':
 * @property integer $id
 * @property string $part_number
 * @property string $bill_of_entry
 * @property integer $from_warehouse
 * @property integer $to_warehouse
 * @property string $quantity
 * @property integer $from_stock_id
 * @property string $notes
 * @property integer $created_by
 * @property string $created_datetime
 */
class WarehouseTransferHistory extends CActiveRecord
{
	/**
	 * @return string the associated database table name
	 */
	public function tableName()
	{
		return '{{warehouse_transfer_history}}';
	}

	/**
	 * @return array validation rules for model attributes.
	 */
	public $available_quantity;
	public $quantity_unit;
	
	public function rules()
	{
		// NOTE: you should only define rules for those attributes that
		// will receive user inputs.
		return array(
			array('part_number, description, bill_of_entry, from_warehouse, to_warehouse, quantity, from_stock_id, notes, created_by, created_datetime', 'required'),
			array('from_warehouse, to_warehouse, from_stock_id, created_by', 'numerical', 'integerOnly'=>true),
			array('part_number, bill_of_entry', 'length', 'max'=>255),
			array('quantity', 'length', 'max'=>15),
			array('from_stock_id', 'checkExistance'),
			// The following rule is used by search().
			// @todo Please remove those attributes that should not be searched.
			array('id, part_number, description, bill_of_entry, from_warehouse, to_warehouse, quantity, from_stock_id, notes, created_by, created_datetime', 'safe', 'on'=>'search'),
		);
	}

	public function checkExistance($attribute,$params)
    {
        if( $this->from_stock_id > 0 )
        {
            $model = Stock::model()->findBypk( $this->from_stock_id  );
            if( $model->quantity < $this->quantity )
                $this->addError('id','Item is not available in selected Location');
        }   
    } 

	/**
	 * @return array relational rules.
	 */
	public function relations()
	{
		// NOTE: you may need to adjust the relation name and the related
		// class name for the relations automatically generated below.
		return array(
			'fromwarehouseDetails' => array(self::BELONGS_TO, 'Warehouses', 'from_warehouse'),
			'towarehouseDetails' => array(self::BELONGS_TO, 'Warehouses', 'to_warehouse'),
			'fromStock' => array(self::BELONGS_TO, 'Stock', 'from_stock_id'),
			'inboundRelation' => array(self::BELONGS_TO, 'Inbound', 'inbound_id'),
		);
	}

	/**
	 * @return array customized attribute labels (name=>label)
	 */
	public function attributeLabels()
	{
		return array(
			'id' => 'ID',
			'part_number' => 'Part Number',
			'bill_of_entry' => 'Bill Of Entry',
			'from_warehouse' => 'From Warehouse',
			'to_warehouse' => 'To Warehouse',
			'quantity' => 'Quantity',
			'from_stock_id' => 'From Stock',
			'notes' => 'Notes',
			'created_by' => 'Created By',
			'created_datetime' => 'Created Datetime',
		);
	}

	/**
	 * Retrieves a list of models based on the current search/filter conditions.
	 *
	 * Typical usecase:
	 * - Initialize the model fields with values from filter form.
	 * - Execute this method to get CActiveDataProvider instance which will filter
	 * models according to data in model fields.
	 * - Pass data provider to CGridView, CListView or any similar widget.
	 *
	 * @return CActiveDataProvider the data provider that can return the models
	 * based on the search/filter conditions.
	 */
	public function search()
	{
		// @todo Please modify the following code to remove attributes that should not be searched.

		$criteria=new CDbCriteria;

		$criteria->compare('id',$this->id);
		$criteria->compare('part_number',$this->part_number,true);
		$criteria->compare('bill_of_entry',$this->bill_of_entry,true);
		$criteria->compare('from_warehouse',$this->from_warehouse);
		$criteria->compare('to_warehouse',$this->to_warehouse);
		$criteria->compare('quantity',$this->quantity,true);
		$criteria->compare('from_stock_id',$this->from_stock_id);
		$criteria->compare('notes',$this->notes,true);
		$criteria->compare('created_by',$this->created_by);
		$criteria->compare('created_datetime',$this->created_datetime,true);

		return new CActiveDataProvider($this, array(
			'criteria'=>$criteria,
		));
	}

	/**
	 * Returns the static model of the specified AR class.
	 * Please note that you should have this exact method in all your CActiveRecord descendants!
	 * @param string $className active record class name.
	 * @return WarehouseTransferHistory the static model class
	 */
	public static function model($className=__CLASS__)
	{
		return parent::model($className);
	}

	public function fill_stock_details(){
		if( $this->from_stock_id > 0 ){
			$stockModel = Stock::model()->findBypk( $this->from_stock_id ); 
			$this->part_number = $stockModel->part_number;
			$this->description = $stockModel->description;
			$this->bill_of_entry = $stockModel->bill_of_entry;
		}
	}
}
